function spring_mass_running_return_map
%% Tabula Rasa

clear all
close all
clc

%% Parameters

l0 = 1;		%uncompressed spring length [m]
m = 80;		%mass [kg]
k = 23000;		%spring stiffness [N/m]
g = 9.81;		%acceleration due to gravity [m/s^2]
alpha0 = 70;	%landing angle of leg [deg]

alpha0 = alpha0*pi/180;

%% Initial Conditions

x = 0;
y = 1;
x_dot = 5;
y_dot = 0;

x1 = -l0*cos(alpha0);

%% State variable
traj = [];
y_apex = [];

%% Transition Conditions

foot_impact = odeset('Events', @touch_down);
push_off = odeset('Events', @take_off);
peak = odeset('Events', @apex);

%% Simulation
for y = 0.94:0.01:1.8
    traj = [];
    q = [x y x_dot y_dot];
    [~,q] = ode45(@EoM_air, [0,5], q,foot_impact);
    traj = [traj; q];
    q = [x1 traj(end,2:4)]';
    [~,q] = ode45(@EoM_ground, [0,5], q,push_off);
    q(:,1) = q(:,1) + traj(end,1) + abs(x1);
    traj = [traj; q];
    q = traj(end,:);
    [~,q] = ode45(@EoM_air, [0,5], q,peak);
    traj = [traj; q];
    y_apex = [y_apex; traj(1,2) traj(end,2)];
end

%%

plot(y_apex(:,1),y_apex(:,2),'r')
axis([0.9 1.2 0.9 1.2])
line([0.9 1.2], [0.9 1.2])
title('Apex to apex return map of spring mass running')
xlabel('y^{apex}_i (m)')
ylabel('y^{apex}_{i+1} (m)')

%% Output Figure - uncomment to print a .eps file.
% print spring_mass_running_return_map.eps -depsc2

%% Equation of Motion for Flight Phase
function output = EoM_air(~,q)
    output(1,1) = q(3);
    output(2,1) = q(4);
    output(3,1) = 0;
	output(4,1) = -g;
end
%% Equation of Motion for Stance Phase
function output = EoM_ground(~,q)
    omega = sqrt(k/m);
	output(1,1) = q(3);
	output(2,1) = q(4);
	output(3,1) = q(1)*omega^2*(l0/sqrt(q(1)^2 + q(2)^2) - 1);
	output(4,1) = q(2)*omega^2*(l0/sqrt(q(1)^2 + q(2)^2) - 1) - g;
end
%% Conditions for step completion
function [impact,terminate,direction] = apex(~,q)
    impact = q(4);
	terminate = 1;
	direction = -1;
end
%% Conditions for take off
function [impact,terminate,direction] = take_off(~,q)
    impact = sqrt(q(1)^2 + q(2)^2) - l0;
	terminate = 1;
	direction = 1;
end
%% Conditions for touch down
function [impact,terminate,direction] = touch_down(~,q)
	impact = q(2) - l0*sin(alpha0);
	terminate = 1;
	direction = -1;
end
end